<?php

/**
 * Plugin Name: AutoQuill
 * Plugin URI:  https://github.com/plusclouds/AutoQuill
 * Description: Fetches blog posts from the PlusClouds API using a Bearer Token and displays them in WordPress admin with search, pagination, AI content generation, and save-to-draft functionality.
 * Version:     1.5
 * Author:      PlusClouds
 * Author URI:  https://plusclouds.com
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: autoquill
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) exit;

class API_Data_Fetcher {
    private $api_base_url = 'https://apiv4.plusclouds.com/blogs/posts-perspective';
    private $preview_api_url = 'https://apiv4.plusclouds.com/blogs/posts/';
    private $generate_api_url = 'https://apiv4.plusclouds.com/ai/blogs/posts/auto-quill-post-generator/auto-quill';
    private $token;

    public function __construct() {
        add_action('admin_menu', [$this, 'init_menu']);
        add_action('admin_post_save_api_settings', [$this, 'handle_settings']);
        add_action('admin_post_save_api_post', [$this, 'handle_save_draft']);
        add_action('wp_ajax_fetch_post_preview', [$this, 'ajax_preview']);
        add_action('wp_ajax_generate_ai_content', [$this, 'ajax_generate']);
        add_action('admin_enqueue_scripts', [$this, 'load_assets']);
        add_action('wp_ajax_aq_auth_action', [$this, 'aq_handle_auth_api']);
        add_action('wp_head', function() {
            if (is_singular()) {
                $post_id = get_the_ID();

                $desc  = get_post_meta($post_id, '_aq_meta_description', true);
                $keys  = get_post_meta($post_id, '_aq_meta_keywords', true);
                $title = get_post_meta($post_id, '_aq_meta_title', true);

                // Yoast SEO yüklüyse description/keywords çıkışını Yoast'a bırak
                if (!defined('WPSEO_VERSION')) {
                    if ($desc) {
                        echo '<meta name="description" content="' . esc_attr($desc) . '">' . PHP_EOL;
                    }
                    if ($keys) {
                        echo '<meta name="keywords" content="' . esc_attr($keys) . '">' . PHP_EOL;
                    }
                }

                // OG başlık her durumda yazılır (Yoast da genellikle bunun üzerine yazmaz)
                if ($title) {
                    echo '<meta property="og:title" content="' . esc_attr($title) . '">' . PHP_EOL;
                }
            }
        });

        // aq_token kontrolü (Option ismini tutarlı yapalım: aq_token)
        $this->token = get_option('aq_token', '');
    }

    public function init_menu() {
        $db_token = get_option('aq_token', '');
        $start_page = !empty($db_token) ? [$this, 'render_list'] : [$this, 'render_welcome'];

        add_menu_page('AutoQuill', 'AutoQuill', 'manage_options', 'api-data', $start_page, 'dashicons-robot', 20);
        add_submenu_page('api-data', __('Start now', 'autoquill'), __('Start now', 'autoquill'), 'manage_options', 'autoquill-welcome', [$this, 'render_welcome']);
        add_submenu_page('api-data', __('Pricing Plans', 'autoquill'), __('Pricing Plans', 'autoquill'), 'manage_options', 'plans', [$this, 'render_plans']);
        add_submenu_page('api-data', __('Support Hub', 'autoquill'), __('Support Hub', 'autoquill'), 'manage_options', 'autoquill-support', [$this, 'render_support']);
        add_submenu_page(null, __('Preview', 'autoquill'), __('Preview', 'autoquill'), 'manage_options', 'api-data-preview', [$this, 'render_preview']);
        add_submenu_page(null, __('Generate', 'autoquill'), __('Generate', 'autoquill'), 'manage_options', 'api-data-new', [$this, 'render_workspace']);
    }

    public function load_assets() {
        $base = plugin_dir_url(__FILE__) . 'assets/js/';
        wp_enqueue_script('aq-tailwind', $base . 'tailwind.min.js', [], '4', false);
        wp_enqueue_script('aq-tailwind-typography', $base . 'tailwind-typography.min.js', ['aq-tailwind'], '4', false);
        wp_enqueue_script('marked-js', $base . 'marked.min.js', [], '15', true);
        wp_localize_script('jquery', 'aq_vars', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce'    => wp_create_nonce('aq_nonce'),
                'loading_text' => __('ENGINE RUNNING...', 'autoquill'),
                'status_generating' => __('GENERATING', 'autoquill'),
                'status_ready' => __('READY', 'autoquill'),
                'generate_btn_text' => __('GENERATE ✨', 'autoquill'),
                'please_wait' => __('PLEASE WAIT...', 'autoquill'),
                'something_went_wrong' => __('Something went wrong.', 'autoquill'),
                'email_check_failed' => __('Email check failed.', 'autoquill'),
                'invalid_expired_code' => __('Invalid or expired code.', 'autoquill'),
                'please_enter_token' => __('Please enter a token.', 'autoquill'),
                'invalid_token' => __('Invalid token.', 'autoquill'),
                'error_occurred' => __('An error occurred.', 'autoquill'),
        ]);
    }

    public function render_list() {
        if (empty($this->token)) {
            echo '<script>window.location.href="' . esc_url(admin_url('admin.php?page=autoquill-welcome')) . '";</script>';
            exit;
        }

        $search = isset($_GET['search']) ? sanitize_text_field(wp_unslash($_GET['search'])) : '';
        $locale = isset($_GET['aq_locale']) ? sanitize_text_field(wp_unslash($_GET['aq_locale'])) : '';
        $page = isset($_GET['paged']) ? max(1, intval(wp_unslash($_GET['paged']))) : 1;

        $languages = [
                ['code' => 'tr', 'name' => 'Türkçe', 'flag' => '🇹🇷'],
                ['code' => 'en', 'name' => 'English', 'flag' => '🇺🇸'],
                ['code' => 'nl', 'name' => 'Nederlands', 'flag' => '🇳🇱'],
                ['code' => 'ar', 'name' => 'العربية', 'flag' => '🇸🇦']
        ];

        $url = $this->api_base_url . '?page=' . $page;
        if (!empty($search)) $url .= '&title=' . urlencode($search);
        if (!empty($locale)) $url .= '&locale=' . urlencode($locale);

        $res = wp_remote_get($url, [
                'headers' => [
                        'Authorization' => 'Bearer ' . $this->token,
                        'Accept' => 'application/json'
                ],
                'timeout' => 20
        ]);

        $body = wp_remote_retrieve_body($res);
        $data = json_decode($body, true);

        echo '<style>@import url("https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap"); .font-corp{font-family:"Inter",sans-serif;} .line-clamp-2{display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;overflow:hidden;} .line-clamp-3{display:-webkit-box;-webkit-line-clamp:3;-webkit-box-orient:vertical;overflow:hidden;}</style>';
        ?>

        <?php $this->render_plan_badge(); ?>

        <div class="p-6 bg-[#f8fafc] min-h-screen font-corp text-slate-800">
            <div class="max-w-[1400px] mx-auto">

                <div class="flex flex-col lg:flex-row justify-between items-end gap-12 mb-16 border-b border-slate-200 pb-12">
                    <div class="max-w-4xl">
                        <div class="inline-flex items-center gap-2 px-3 py-1.5 rounded-full bg-indigo-50 border border-indigo-100 text-indigo-600 text-[10px] font-black uppercase tracking-[0.2em] italic mb-6">
                            <span class="relative flex h-2 w-2">
                                <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-indigo-400 opacity-75"></span>
                                <span class="relative inline-flex rounded-full h-2 w-2 bg-indigo-500"></span>
                            </span>
                            <?php esc_html_e('Active Infrastructure', 'autoquill'); ?>
                        </div>

                        <h1 class="text-5xl md:text-7xl font-extrabold tracking-tighter text-slate-900 mb-6 uppercase italic leading-[0.9]">
                            <?php esc_html_e('Content', 'autoquill'); ?><br>
                            <span class="text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 to-violet-600"><?php esc_html_e('Assets', 'autoquill'); ?></span>
                        </h1>

                        <div class="space-y-4">
                            <p class="text-slate-400 font-bold text-xs uppercase tracking-[0.3em]">
                                <?php esc_html_e('The ultimate AI engine for SEO-optimized content.', 'autoquill'); ?>
                            </p>
                            <p class="text-lg text-slate-500 font-medium leading-relaxed italic max-w-2xl">
                                <?php esc_html_e('Automate your SEO operations and build digital assets at scale with Google E-E-A-T compliant intelligence.', 'autoquill'); ?>
                            </p>
                        </div>
                    </div>

                    <form method="get" class="w-full lg:w-auto">
                        <input type="hidden" name="page" value="api-data">
                        <div class="flex items-center bg-white p-2.5 rounded-[1.5rem] border border-slate-200 shadow-2xl">
                            <select name="aq_locale" onchange="this.form.submit()" class="bg-slate-50 border-none rounded-xl px-4 py-3 text-[10px] font-black uppercase tracking-widest text-slate-600 focus:ring-0 cursor-pointer italic outline-none">
                                <option value=""><?php echo esc_html(__('All Systems', 'autoquill')); ?></option>
                                <?php foreach ($languages as $lang): ?>
                                    <option value="<?php echo esc_attr($lang['code']); ?>" <?php echo selected($locale, $lang['code']); ?>>
                                        <?php echo esc_html($lang['flag'] . ' ' . $lang['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>

                            <div class="h-8 w-px bg-slate-100 mx-2"></div>

                            <input type="text" name="search" value="<?php echo esc_attr($search); ?>" class="bg-transparent border-none py-2 px-4 text-sm font-bold text-slate-700 placeholder:text-slate-300 focus:ring-0 min-w-[200px] outline-none" placeholder="<?php esc_attr_e('Search keywords...', 'autoquill'); ?>">

                            <button type="submit" class="bg-indigo-900 text-white px-8 py-4 rounded-xl text-[10px] font-black uppercase tracking-[0.2em] hover:bg-black transition-all italic shadow-lg">
                                <?php echo esc_html(__('Search', 'autoquill')); ?>
                            </button>
                        </div>
                    </form>
                </div>

                <div id="aq-welcome-bar" class="hidden mb-6 bg-white rounded-lg p-4 border border-slate-200">
                    <div class="flex justify-between items-center gap-4">
                        <div>
                            <p class="text-sm font-medium text-slate-700"><?php esc_html_e('Ready to grow your blog?', 'autoquill'); ?></p>
                            <p class="text-xs text-slate-500 mt-0.5"><?php esc_html_e('Select a post, hit generate, and watch the magic happen. Your next masterpiece is just a click away.', 'autoquill'); ?></p>
                        </div>
                        <button onclick="closeWelcomeBar()" class="text-slate-400 hover:text-slate-600 p-1 shrink-0">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                        </button>
                    </div>
                </div>

                <script>
                    document.addEventListener('DOMContentLoaded', function() {
                        if (!localStorage.getItem('aq_welcome_closed')) {
                            document.getElementById('aq-welcome-bar').classList.remove('hidden');
                        }
                    });
                    function closeWelcomeBar() {
                        document.getElementById('aq-welcome-bar').classList.add('hidden');
                        localStorage.setItem('aq_welcome_closed', 'true');
                    }
                </script>

                <?php if (!empty($data['data'])): ?>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                        <?php foreach ($data['data'] as $item): ?>
                            <div class="bg-white rounded-lg p-5 border border-slate-200 hover:border-indigo-300 transition flex flex-col justify-between">
                                <div class="mb-4">
                                    <h3 class="text-base font-semibold text-slate-800 mb-2 line-clamp-2 min-h-[2.75rem]">
                                        <?php echo esc_html($item['title']); ?>
                                    </h3>
                                    <?php if(!empty($item['abstract'])): ?>
                                        <p class="text-slate-500 text-sm leading-relaxed line-clamp-3">
                                            <?php echo esc_html($item['abstract']); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                                <div class="pt-4 border-t border-slate-100 flex items-center justify-between gap-2">
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=api-data-preview&id='.$item['id'])); ?>"
                                       class="text-slate-400 hover:text-indigo-600 transition p-1" title="Preview">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/></svg>
                                    </a>
                                    <div class="flex gap-2">
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=api-data-new&id='.$item['id'])); ?>"
                                           class="text-slate-600 px-3 py-1.5 rounded-md text-xs font-medium hover:bg-slate-100 transition border border-slate-200">
                                            <?php echo esc_html(__('Generate', 'autoquill')); ?>
                                        </a>
                                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="m-0">
                                            <input type="hidden" name="action" value="save_api_post">
                                            <input type="hidden" name="blog_post_id" value="<?php echo esc_attr($item['id']); ?>">
                                            <?php wp_nonce_field('aq_save_post', 'aq_save_nonce'); ?>
                                            <button type="submit" class="bg-indigo-900 text-white px-4 py-2 rounded-xl text-[10px] font-black uppercase tracking-widest hover:bg-indigo-900 transition-all shadow-lg italic">
                                                <?php echo esc_html(__('Quick Save', 'autoquill')); ?>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="bg-white rounded-lg p-16 text-center border border-slate-200">
                        <p class="text-lg text-slate-500"><?php echo esc_html(__('No content found.', 'autoquill')); ?></p>
                    </div>
                <?php endif; ?>
            </div>

            <?php $this->render_footer(); ?>
        </div>

        <?php
    }

    public function render_preview() {
        $id = sanitize_text_field(wp_unslash($_GET['id'] ?? ''));
        ?>
        <style>
            @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');
            .font-corp { font-family: 'Inter', sans-serif; }
            .prose { max-width: 100% !important; line-height: 1.75; color: #334155; }
            .prose h2, .prose h3 { color: #0f172a; font-weight: 700; }
            .prose img { border-radius: 0.5rem; }
        </style>

        <?php $this->render_plan_badge(); ?>

        <div class="p-6 bg-[#f8fafc] min-h-screen font-corp">
            <div class="max-w-6xl mx-auto">
                <div class="flex justify-between items-center mb-5">
                    <a href="?page=api-data" class="flex items-center gap-2 text-slate-500 hover:text-indigo-600 transition text-sm font-medium">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path d="M15 19l-7-7 7-7" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>
                        <?php esc_html_e('Back to List', 'autoquill'); ?>
                    </a>
                    <span class="text-xs text-slate-400"><?php esc_html_e('Analyzed: 100% Human-Like Score', 'autoquill'); ?></span>
                </div>

                <div class="flex flex-col lg:flex-row gap-6">
                    <div class="w-full lg:flex-1">
                        <div class="bg-white rounded-lg border border-slate-200 overflow-hidden relative">
                            <div id="pv-loader" class="absolute inset-0 bg-white z-50 flex items-center justify-center">
                                <div class="flex flex-col items-center gap-3">
                                    <div class="w-8 h-8 border-3 border-slate-200 border-t-indigo-600 rounded-full animate-spin"></div>
                                    <span class="text-xs text-slate-400"><?php esc_html_e('Preparing Content...', 'autoquill'); ?></span>
                                </div>
                            </div>
                            <div id="pv-img-wrap" class="hidden w-full h-[240px] relative">
                                <img id="pv-header-image" src="" class="w-full h-full object-cover">
                            </div>
                            <div class="p-8">
                                <h1 id="pv-title" class="text-2xl font-bold text-slate-900 leading-tight mb-6"></h1>
                                <article id="pv-body" class="prose prose-slate prose-sm border-t border-slate-100 pt-6"></article>
                            </div>
                        </div>
                    </div>

                    <div class="w-full lg:w-[400px] space-y-6">

                        <div class="bg-white rounded-[3rem] p-10 border border-slate-200 shadow-[0_30px_60px_-15px_rgba(0,0,0,0.05)] relative overflow-hidden group">
                            <div class="absolute top-0 right-0 w-32 h-32 bg-indigo-50 rounded-bl-[5rem] -mr-10 -mt-10 transition-transform group-hover:scale-110"></div>
                            <div class="relative z-10">
                                <div class="flex items-center gap-2 mb-8">
                                    <span class="w-2 h-2 bg-indigo-600 rounded-full"></span>
                                    <span class="text-[10px] font-black text-indigo-600 uppercase tracking-[0.2em] italic"><?php esc_html_e('Draft Configuration', 'autoquill'); ?></span>
                                </div>
                                <h3 class="text-3xl font-black text-slate-900 uppercase italic tracking-tighter leading-none mb-4">
                                    <?php esc_html_e('Ready to', 'autoquill'); ?><br>
                                    <span class="text-indigo-600"><?php esc_html_e('Deploy?', 'autoquill'); ?></span>
                                </h3>
                                <p class="text-slate-500 text-xs font-bold uppercase italic tracking-tight leading-relaxed mb-8 opacity-80">
                                    <?php esc_html_e('Convert this intelligence into a high-authority digital asset with one click.', 'autoquill'); ?>
                                </p>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=api-data-new&id='.$id)); ?>" class="flex items-center hover:text-white justify-center w-full bg-indigo-950 text-white py-6 rounded-2xl text-[11px] font-black uppercase tracking-[0.2em] hover:bg-black transition-all shadow-xl shadow-indigo-900/20 italic mb-4">
                                    <?php esc_html_e('🚀 Generate Content', 'autoquill'); ?>
                                </a>
                                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="m-0">
                                    <input type="hidden" name="action" value="save_api_post">
                                    <input type="hidden" name="blog_post_id" value="<?php echo esc_attr($id); ?>">
                                    <?php wp_nonce_field('aq_save_post', 'aq_save_nonce'); ?>
                                    <button type="submit" class="w-full py-4 bg-slate-50 text-slate-400 border border-slate-100 rounded-2xl text-[10px] font-black uppercase tracking-widest hover:text-indigo-600 hover:bg-white hover:border-indigo-100 transition-all italic">
                                        <?php esc_html_e('Save to Drafts', 'autoquill'); ?>
                                    </button>
                                </form>
                            </div>
                        </div>

                        <div class="bg-indigo-950 rounded-[3rem] p-10 text-white shadow-2xl shadow-indigo-950/40 relative overflow-hidden">
                            <div class="absolute -left-10 -bottom-10 w-40 h-40 bg-indigo-500 opacity-10 rounded-full blur-3xl"></div>
                            <div class="relative z-10">
                                <h4 class="text-[10px] font-black text-indigo-400 mb-8 uppercase tracking-[0.3em] italic"><?php esc_html_e('Core Engine Specs', 'autoquill'); ?></h4>
                                <div class="space-y-8">
                                    <div class="flex items-center gap-5">
                                        <div class="flex-shrink-0 w-10 h-10 bg-white/5 rounded-xl flex items-center justify-center border border-white/10 transition-colors">
                                            <svg class="w-5 h-5 text-indigo-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/></svg>
                                        </div>
                                        <div>
                                            <p class="text-[11px] font-black uppercase italic tracking-wide"><?php esc_html_e('EEAT Optimized', 'autoquill'); ?></p>
                                            <p class="text-[10px] text-indigo-300 font-bold uppercase opacity-50"><?php esc_html_e('Search Engine Authority', 'autoquill'); ?></p>
                                        </div>
                                    </div>
                                    <div class="flex items-center gap-5">
                                        <div class="flex-shrink-0 w-10 h-10 bg-white/5 rounded-xl flex items-center justify-center border border-white/10">
                                            <svg class="w-5 h-5 text-indigo-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>
                                        </div>
                                        <div>
                                            <p class="text-[11px] font-black uppercase italic tracking-wide"><?php esc_html_e('Anti-AI Guard', 'autoquill'); ?></p>
                                            <p class="text-[10px] text-indigo-300 font-bold uppercase opacity-50"><?php esc_html_e('Undetectable Output', 'autoquill'); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="bg-slate-100/50 rounded-[3rem] p-10 border border-dashed border-slate-300 relative overflow-hidden hover:border-indigo-300 transition-all group">
                            <div class="relative z-10">
                                <h4 class="text-[10px] font-black text-slate-400 mb-4 uppercase tracking-[0.3em] italic"><?php esc_html_e('Not Feeling This Topic?', 'autoquill'); ?></h4>
                                <p class="text-slate-500 text-[11px] font-bold uppercase italic mb-6 leading-relaxed">
                                    <?php esc_html_e('Explore hundreds of other high-potential topics waiting in your library.', 'autoquill'); ?>
                                </p>
                                <a href="?page=api-data" class="inline-flex items-center gap-3 text-indigo-600 group-hover:text-indigo-700 text-[11px] font-black uppercase tracking-widest italic transition-all">
                                    <span><?php esc_html_e('Return to Library', 'autoquill'); ?></span>
                                    <svg class="w-4 h-4 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path d="M17 8l4 4m0 0l-4 4m4-4H3" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"/></svg>
                                </a>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </div>

        <?php $this->render_footer(); ?>

        <script>
            jQuery(document).ready(function($){
                $.post(aq_vars.ajax_url, {
                    action: 'fetch_post_preview',
                    nonce: aq_vars.nonce,
                    post_id: "<?php echo esc_attr($id); ?>"
                }, function(r) {
                    $('#pv-loader').fadeOut(200);
                    if(r.success && r.data) {
                        if(r.data.header_image) {
                            $('#pv-header-image').attr('src', r.data.header_image);
                            $('#pv-img-wrap').show();
                        }
                        $('#pv-title').text(r.data.title || '');
                        $('#pv-body').html(r.data.body ? marked.parse(r.data.body) : '');
                    }
                });
            });
        </script>
        <?php
    }

    public function render_plans() {
        ?>
        <style>
            @import url('https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@300;400;500;600;700;800&display=swap');
            .font-modern { font-family: 'Plus Jakarta Sans', sans-serif; }
            .pro-shadow { box-shadow: 0 25px 50px -12px rgba(79, 70, 229, 0.2); }
            .accent-gradient { background: linear-gradient(90deg, #4f46e5 0%, #7c3aed 100%); }
        </style>

        <?php $this->render_plan_badge(); ?>

        <div class="min-h-screen bg-[#f8fafc] p-6 md:p-16 lg:p-24 font-modern text-slate-900">
            <div class="max-w-7xl mx-auto">

                <div class="max-w-3xl mb-16">
                    <div class="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-indigo-50 border border-indigo-100 text-indigo-600 text-[11px] font-bold uppercase tracking-widest mb-6">
                    <span class="relative flex h-2 w-2">
                        <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-indigo-400 opacity-75"></span>
                        <span class="relative inline-flex rounded-full h-2 w-2 bg-indigo-500"></span>
                    </span>
                        <?php esc_html_e('Enterprise Ready', 'autoquill'); ?>
                    </div>
                    <h1 class="text-4xl md:text-6xl font-extrabold tracking-tight text-slate-900 leading-[1.1] mb-6">
                        <?php esc_html_e('Next-Gen Content', 'autoquill'); ?><br/>
                        <span class="text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 to-violet-600"><?php esc_html_e('Infrastructure', 'autoquill'); ?></span>
                    </h1>
                    <p class="text-lg text-slate-500 font-medium leading-relaxed">
                        <?php esc_html_e('Automate your SEO operations, reduce costs, and build digital assets at scale with Google E-E-A-T compliant intelligence.', 'autoquill'); ?>
                    </p>
                </div>

                <div class="grid grid-cols-1 lg:grid-cols-12 gap-8 items-stretch">

                    <div class="lg:col-span-4 bg-white rounded-3xl p-10 border border-slate-200 shadow-sm flex flex-col justify-between">
                        <div>
                            <h3 class="text-xl font-bold text-slate-900 mb-2"><?php esc_html_e('Standard Edition', 'autoquill'); ?></h3>
                            <p class="text-slate-400 text-sm mb-8 leading-relaxed">
                                <?php esc_html_e('Essential features for personal testing and basic blog management.', 'autoquill'); ?>
                            </p>

                            <div class="mb-8">
                                <span class="text-4xl font-bold text-slate-900 tracking-tight">$0</span>
                                <span class="text-slate-400 font-medium">/ mo</span>
                            </div>

                            <ul class="space-y-4 mb-8 text-sm">
                                <li class="flex items-center gap-3 text-slate-400 line-through decoration-slate-300">
                                    <svg class="w-5 h-5 shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                                    <?php esc_html_e('Advanced AI Models', 'autoquill'); ?>
                                </li>
                                <li class="flex items-center gap-3 text-slate-600 font-medium">
                                    <svg class="w-5 h-5 text-indigo-500 shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M5 13l4 4L19 7"/></svg>
                                    <?php esc_html_e('GPT-3.5 Standard Engine', 'autoquill'); ?>
                                </li>
                                <li class="flex items-center gap-3 text-slate-600 font-medium">
                                    <svg class="w-5 h-5 text-indigo-500 shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M5 13l4 4L19 7"/></svg>
                                    <?php esc_html_e('Standard Support', 'autoquill'); ?>
                                </li>
                            </ul>
                        </div>

                        <a href="<?php echo esc_url(admin_url('admin.php?page=api-data')); ?>" class="w-full py-4 bg-slate-50 hover:bg-slate-100 text-slate-600 text-center text-xs font-bold uppercase tracking-widest rounded-xl border border-slate-200 transition-colors">
                            <?php esc_html_e('Current Plan', 'autoquill'); ?>
                        </a>
                    </div>

                    <div class="lg:col-span-8 bg-slate-950 rounded-3xl overflow-hidden relative pro-shadow flex flex-col md:flex-row">
                        <div class="absolute top-0 right-0 w-64 h-64 accent-gradient opacity-20 blur-[100px] -mr-20 -mt-20"></div>

                        <div class="p-10 md:p-14 flex-1 relative z-10 border-r border-white/5">
                            <div class="flex items-center gap-3 mb-6">
                            <span class="px-3 py-1 bg-white/10 text-white text-[10px] font-bold uppercase tracking-tighter rounded-md border border-white/10">
                                <?php esc_html_e('Recommended for Agencies', 'autoquill'); ?>
                            </span>
                            </div>
                            <h3 class="text-3xl font-bold text-white mb-4"><?php esc_html_e('AutoQuill Professional', 'autoquill'); ?></h3>
                            <p class="text-slate-400 text-sm leading-relaxed mb-10 max-w-md">
                                <?php esc_html_e('Engineered for scale. High-performance models with anti-detect technology and semantic optimization.', 'autoquill'); ?>
                            </p>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-10 gap-y-5">
                                <?php
                                $pro_features = [
                                        __('Global Multi-Language Support', 'autoquill'),
                                        __('Perplexity & Llama-3 Access', 'autoquill'),
                                        __('White-Label Content', 'autoquill'),
                                        __('E-E-A-T Semantic Engine', 'autoquill'),
                                        __('Bypass AI Detectors', 'autoquill'),
                                        __('Enterprise SLA & Priority API', 'autoquill')
                                ];
                                foreach($pro_features as $feature): ?>
                                    <div class="flex items-center gap-3 text-slate-200 text-sm font-medium">
                                        <div class="p-0.5 rounded-full bg-indigo-500/20 text-indigo-400">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M5 13l4 4L19 7"/></svg>
                                        </div>
                                        <?php echo esc_html($feature); ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <div class="p-10 md:w-80 flex flex-col justify-center items-center bg-white/[0.02] backdrop-blur-md relative z-10">
                            <div class="text-center mb-10">
                                <span class="block text-slate-500 text-[11px] font-bold uppercase tracking-[0.2em] mb-2"><?php esc_html_e('Monthly License', 'autoquill'); ?></span>
                                <div class="flex items-baseline justify-center gap-1">
                                    <span class="text-6xl font-extrabold text-white tracking-tighter">$7.99</span>
                                </div>
                                <p class="mt-4 text-[11px] text-emerald-400 font-bold uppercase tracking-wider">
                                    <?php esc_html_e('Instant Activation', 'autoquill'); ?>
                                </p>
                            </div>

                            <a href="https://plusclouds.com/us/community/auto-quill#pricing" target="_blank" class="w-full hover:text-white py-5 accent-gradient hover:opacity-90 text-white text-center text-xs font-black uppercase tracking-widest rounded-2xl transition-all shadow-2xl shadow-indigo-500/40 transform hover:-translate-y-1">
                                <?php esc_html_e('Upgrade to Pro', 'autoquill'); ?>
                            </a>

                            <p class="mt-6 text-[10px] text-slate-500 font-medium uppercase tracking-tight">
                                <?php esc_html_e('Cancel anytime', 'autoquill'); ?>
                            </p>
                        </div>
                    </div>
                </div>

                <div class="mt-24 pt-16 border-t border-slate-200 grid grid-cols-1 md:grid-cols-3 gap-12 text-center md:text-left">
                    <div>
                        <div class="text-indigo-600 mb-4 font-black text-2xl tracking-tighter italic font-serif leading-none">01.</div>
                        <h4 class="text-sm font-bold text-slate-900 uppercase tracking-widest mb-3"><?php esc_html_e('Compliance', 'autoquill'); ?></h4>
                        <p class="text-sm text-slate-500 leading-relaxed font-medium">
                            <?php esc_html_e('Every data set is checked against Google Search Quality Guidelines to ensure long-term domain authority.', 'autoquill'); ?>
                        </p>
                    </div>
                    <div>
                        <div class="text-indigo-600 mb-4 font-black text-2xl tracking-tighter italic font-serif leading-none">02.</div>
                        <h4 class="text-sm font-bold text-slate-900 uppercase tracking-widest mb-3"><?php esc_html_e('Global Reach', 'autoquill'); ?></h4>
                        <p class="text-sm text-slate-500 leading-relaxed font-medium">
                            <?php esc_html_e('Deploy native-level content in 50+ languages, maintaining semantic integrity across global markets.', 'autoquill'); ?>
                        </p>
                    </div>
                    <div>
                        <div class="text-indigo-600 mb-4 font-black text-2xl tracking-tighter italic font-serif leading-none">03.</div>
                        <h4 class="text-sm font-bold text-slate-900 uppercase tracking-widest mb-3"><?php esc_html_e('Performance', 'autoquill'); ?></h4>
                        <p class="text-sm text-slate-500 leading-relaxed font-medium">
                            <?php esc_html_e('High-volume content pipelines designed for rapid SEO execution and zero algorithmic penalties.', 'autoquill'); ?>
                        </p>
                    </div>
                </div>

                <div class="mt-20 text-center">
                    <a href="<?php echo esc_url(admin_url('admin.php?page=autoquill-support')); ?>" class="group inline-flex items-center gap-2 text-sm font-bold text-slate-400 hover:text-indigo-600 transition-colors">
                        <?php esc_html_e('Need custom enterprise solutions? Contact our specialists', 'autoquill'); ?>
                        <svg class="w-4 h-4 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"/></svg>
                    </a>
                </div>

                <?php $this->render_footer(); ?>
            </div>
        </div>
        <?php
    }

    public function render_support() {
        ?>
        <style>
            @import url('https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@300;400;500;600;700;800&display=swap');
            .font-modern { font-family: 'Plus Jakarta Sans', sans-serif; }
            .support-card { transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1); }
            .support-card:hover { transform: translateY(-8px); }
            .accent-gradient { background: linear-gradient(90deg, #4f46e5 0%, #7c3aed 100%); }
        </style>

        <?php $this->render_plan_badge(); ?>

        <div class="min-h-screen bg-[#f8fafc] p-6 md:p-16 lg:p-24 font-modern text-slate-900">
            <div class="max-w-7xl mx-auto">

                <div class="max-w-3xl mb-16">
                    <div class="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-indigo-50 border border-indigo-100 text-indigo-600 text-[11px] font-bold uppercase tracking-widest mb-6">
                        <?php esc_html_e('Support & Community', 'autoquill'); ?>
                    </div>
                    <h1 class="text-4xl md:text-6xl font-extrabold tracking-tight text-slate-900 leading-[1.1] mb-6">
                        <?php esc_html_e('How can we help', 'autoquill'); ?><br/>
                        <span class="text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 to-violet-600"><?php esc_html_e('your growth?', 'autoquill'); ?></span>
                    </h1>
                    <p class="text-lg text-slate-500 font-medium leading-relaxed">
                        <?php esc_html_e('Access technical documentation, join our elite community of SEO professionals, or reach out for enterprise-level inquiries.', 'autoquill'); ?>
                    </p>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-3 gap-8 mb-20">

                    <div class="support-card bg-white rounded-3xl p-10 border border-slate-200 shadow-sm flex flex-col items-center text-center">
                        <div class="w-16 h-16 bg-[#5865F2]/10 text-[#5865F2] rounded-2xl flex items-center justify-center mb-8">
                            <svg class="w-8 h-8" fill="currentColor" viewBox="0 0 24 24"><path d="M20.317 4.37a19.791 19.791 0 0 0-4.885-1.515a.074.074 0 0 0-.079.037c-.21.375-.444.864-.608 1.25a18.27 18.27 0 0 0-5.487 0a12.64 12.64 0 0 0-.617-1.25a.077.077 0 0 0-.079-.037A19.736 19.736 0 0 0 3.677 4.37a.07.07 0 0 0-.032.027C.533 9.046-.32 13.58.099 18.057a.082.082 0 0 0 .031.057a19.9 19.9 0 0 0 5.993 3.03a.078.078 0 0 0 .084-.028a14.09 14.09 0 0 0 1.226-1.994a.076.076 0 0 0-.041-.106a13.107 13.107 0 0 1-1.872-.892a.077.077 0 0 1-.008-.128a10.2 10.2 0 0 0 .372-.292a.074.074 0 0 1 .077-.01c3.928 1.793 8.18 1.793 12.062 0a.074.074 0 0 1 .078.01c.12.098.246.198.373.292a.077.077 0 0 1-.006.127a12.299 12.299 0 0 1-1.873.892a.076.076 0 0 0-.041.107c.36.698.772 1.362 1.225 1.993a.076.076 0 0 0 .084.028a19.839 19.839 0 0 0 6.002-3.03a.077.077 0 0 0 .032-.054c.5-5.177-.838-9.674-3.549-13.66a.061.061 0 0 0-.031-.03zM8.02 15.33c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.955-2.419 2.157-2.419c1.21 0 2.176 1.086 2.157 2.419c0 1.334-.956 2.419-2.157 2.419zm7.975 0c-1.183 0-2.157-1.085-2.157-2.419c0-1.333.955-2.419 2.157-2.419c1.21 0 2.176 1.086 2.157 2.419c0 1.334-.946 2.419-2.157 2.419z"/></svg>
                        </div>
                        <h3 class="text-xl font-bold text-slate-900 mb-4"><?php esc_html_e('Discord Hub', 'autoquill'); ?></h3>
                        <p class="text-slate-500 text-sm font-medium mb-10 leading-relaxed">
                            <?php esc_html_e('Connect with top SEO minds and developers in real-time. Shared intelligence at its best.', 'autoquill'); ?>
                        </p>
                        <a href="https://discord.gg/EX4NHQd3" target="_blank" class="w-full py-4 bg-[#5865F2] hover:bg-[#4752c4] hover:text-white text-white rounded-2xl font-bold text-xs uppercase tracking-widest transition-all">
                            <?php esc_html_e('Join Community', 'autoquill'); ?>
                        </a>
                    </div>

                    <div class="support-card bg-white rounded-3xl p-10 border border-slate-200 shadow-sm flex flex-col items-center text-center">
                        <div class="w-16 h-16 bg-indigo-50 text-indigo-600 rounded-2xl flex items-center justify-center mb-8">
                            <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/></svg>
                        </div>
                        <h3 class="text-xl font-bold text-slate-900 mb-4"><?php esc_html_e('Documentation', 'autoquill'); ?></h3>
                        <p class="text-slate-500 text-sm font-medium mb-10 leading-relaxed">
                            <?php esc_html_e('Master our semantic engine with deep-dive tutorials, API guides, and strategic SEO playbooks.', 'autoquill'); ?>
                        </p>
                        <a href="https://plusclouds.com/us/community" target="_blank" class="w-full py-4 bg-indigo-600 hover:text-white hover:bg-indigo-700 text-white rounded-2xl font-bold text-xs uppercase tracking-widest transition-all">
                            <?php esc_html_e('Explore Guides', 'autoquill'); ?>
                        </a>
                    </div>

                    <div class="support-card bg-slate-950 rounded-3xl p-10 flex flex-col items-center text-center relative overflow-hidden">
                        <div class="absolute top-0 right-0 w-32 h-32 accent-gradient opacity-20 blur-[60px] -mr-10 -mt-10"></div>
                        <div class="w-16 h-16 bg-white/5 text-indigo-400 rounded-2xl flex items-center justify-center mb-8 border border-white/10">
                            <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/></svg>
                        </div>
                        <h3 class="text-xl font-bold text-white mb-4"><?php esc_html_e('Direct Support', 'autoquill'); ?></h3>
                        <p class="text-slate-400 text-sm font-medium mb-10 leading-relaxed">
                            <?php esc_html_e('Account, billing or technical assistance. Our specialists respond within 24 business hours.', 'autoquill'); ?>
                        </p>
                        <a href="mailto:info@plusclouds.com" class="w-full py-4 bg-white text-slate-900 hover:bg-slate-100 rounded-2xl font-bold text-xs uppercase tracking-widest transition-all">
                            <?php esc_html_e('Open Ticket', 'autoquill'); ?>
                        </a>
                    </div>
                </div>

                <div class="pt-16 border-t border-slate-200">
                    <div class="grid grid-cols-1 lg:grid-cols-2 gap-16 items-center">
                        <div>
                            <h4 class="text-sm font-bold text-slate-400 uppercase tracking-[0.2em] mb-8 italic">
                                <?php esc_html_e('Trusted by Infrastructure Providers', 'autoquill'); ?>
                            </h4>
                            <div class="flex flex-wrap gap-x-12 gap-y-6 opacity-30 grayscale items-center">
                                <span class="font-black text-2xl tracking-tighter italic">PlusClouds</span>
                                <span class="font-black text-2xl tracking-tighter italic">AutoQuill</span>
                                <span class="font-black text-2xl tracking-tighter italic">AI-Tech</span>
                                <span class="font-black text-2xl tracking-tighter italic">SEO-Core</span>
                            </div>
                        </div>

                        <div class="bg-indigo-50/50 rounded-3xl p-10 border border-indigo-100 relative overflow-hidden group">
                            <div class="relative z-10">
                                <h3 class="text-xl font-bold text-slate-900 mb-4"><?php esc_html_e('Agency & Enterprise Solutions', 'autoquill'); ?></h3>
                                <p class="text-slate-500 text-sm font-medium mb-8 leading-relaxed max-w-md">
                                    <?php esc_html_e('Custom API integrations, high-velocity content pipelines, and white-label licensing for large portfolios.', 'autoquill'); ?>
                                </p>
                                <a href="https://plusclouds.com" target="_blank" class="inline-flex items-center gap-2 text-xs font-bold text-indigo-600 uppercase tracking-widest hover:text-indigo-700 transition-colors">
                                    <?php esc_html_e('Consult with Enterprise Sales', 'autoquill'); ?>
                                    <svg class="w-4 h-4 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3"/></svg>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>

                <?php $this->render_footer(); ?>
            </div>
        </div>
        <?php
    }

    public function render_welcome() {
        $current_token = get_option('aq_token');
        $user_plan = get_option('aq_user_plan', '');
        $active_email = '';

        if ($current_token) {
            $leo_url = "https://apiv4.plusclouds.com";
            $check_res = wp_remote_get("$leo_url/leo/security/am-i-logged-in", [
                    'headers' => ['Authorization' => 'Bearer ' . $current_token, 'Accept' => 'application/json'],
                    'timeout' => 10
            ]);
            if (!is_wp_error($check_res)) {
                $user_body = json_decode(wp_remote_retrieve_body($check_res), true);
                $active_email = $user_body['email'] ?? 'Connected User';
            }
        }

        $wp_users = get_users(array('fields' => array('display_name', 'user_email')));
        $current_user = wp_get_current_user();
        $user_email = $current_user->user_email;
        ?>
        <style>
            @import url('https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@400;500;600;700;800&display=swap');
            .font-modern { font-family: 'Plus Jakarta Sans', sans-serif; }
            .aq-plan-card { transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1); }
            .aq-plan-card:hover { transform: translateY(-5px); }
            .tab-btn.active { background: #4f46e5; color: white; }
        </style>

        <div class="min-h-screen bg-[#f8fafc] p-6 md:p-12 font-modern">
            <div class="max-w-4xl mx-auto text-center mb-12">
                <div class="inline-flex items-center justify-center p-3 bg-indigo-900 rounded-2xl shadow-xl mb-6 rotate-2">
                    <img class="h-12" src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/img/logos/first.png'); ?>">
                </div>

                <h1 class="text-4xl md:text-5xl font-extrabold tracking-tighter text-slate-900 leading-[1] mb-6 uppercase italic">
                    <?php esc_html_e('AutoQuill', 'autoquill'); ?><br/>
                    <span class="text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 to-violet-600">
                        <?php esc_html_e('Next-Gen Infrastructure', 'autoquill'); ?>
                    </span>
                </h1>

                <p class="text-slate-500 font-bold text-xs uppercase tracking-[0.3em]">
                    <?php esc_html_e('The ultimate AI engine for SEO-optimized content.', 'autoquill'); ?>
                </p>
            </div>

            <div class="max-w-5xl mx-auto grid grid-cols-1 md:grid-cols-2 gap-8 items-stretch">

                <div class="aq-plan-card bg-white border border-slate-200 rounded-[2.5rem] p-8 md:p-10 flex flex-col shadow-sm">
                    <div class="mb-8">
        <span class="px-4 py-1.5 rounded-full bg-indigo-50 text-indigo-600 text-[10px] font-black uppercase tracking-widest italic border border-indigo-100">
            <?php esc_html_e('The Multiplier Effect', 'autoquill'); ?>
        </span>
                        <h3 class="text-3xl font-black text-slate-900 mt-6 mb-2 uppercase italic tracking-tighter">
                            <?php esc_html_e('Expert Engine', 'autoquill'); ?>
                        </h3>
                        <p class="text-slate-400 font-bold uppercase text-[10px] tracking-widest italic leading-relaxed">
                            <?php esc_html_e('We don’t fake expertise, we amplify yours.', 'autoquill'); ?>
                        </p>
                    </div>

                    <div class="space-y-5 mb-10 flex-grow">
                        <div class="flex items-start gap-3 text-slate-700">
                            <div class="mt-1 bg-emerald-500 rounded-full p-0.5">
                                <svg class="w-3.5 h-3.5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="4" d="M5 13l4 4L19 7"/></svg>
                            </div>
                            <div class="flex flex-col">
                                <span class="text-[11px] font-black uppercase tracking-widest italic"><?php esc_html_e('Human-First E-E-A-T', 'autoquill'); ?></span>
                                <span class="text-[10px] text-slate-400 font-medium italic"><?php esc_html_e('Designed for SEO clusters and trust.', 'autoquill'); ?></span>
                            </div>
                        </div>

                        <div class="flex items-start gap-3 text-slate-700">
                            <div class="mt-1 bg-emerald-500 rounded-full p-0.5">
                                <svg class="w-3.5 h-3.5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="4" d="M5 13l4 4L19 7"/></svg>
                            </div>
                            <div class="flex flex-col">
                                <span class="text-[11px] font-black uppercase tracking-widest italic"><?php esc_html_e('Voice-to-WP Workflow', 'autoquill'); ?></span>
                                <span class="text-[10px] text-slate-400 font-medium italic"><?php esc_html_e('Starts from your voice, ends on your site.', 'autoquill'); ?></span>
                            </div>
                        </div>

                        <div class="mt-6 p-4 bg-slate-50 rounded-2xl border border-slate-100">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-[9px] font-black text-slate-400 uppercase italic tracking-[0.2em]"><?php esc_html_e('Market vs AutoQuill', 'autoquill'); ?></span>
                                <span class="text-[9px] font-black text-indigo-600 uppercase italic"><?php esc_html_e('Pro Choice', 'autoquill'); ?></span>
                            </div>
                            <div class="space-y-1">
                                <div class="flex justify-between text-[10px] font-bold italic text-slate-600">
                                    <span><?php esc_html_e('Generic AI Writers', 'autoquill'); ?></span>
                                    <span class="text-red-400">$20 - $49/mo</span>
                                </div>
                                <div class="flex justify-between text-[10px] font-black italic text-indigo-900 uppercase">
                                    <span><?php esc_html_e('AutoQuill SEO Multiplier', 'autoquill'); ?></span>
                                    <span><?php esc_html_e('$7.99', 'autoquill'); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <a href="https://plusclouds.com/us/community/auto-quill" target="_blank" class="group block w-full py-5 border-2 border-slate-900 rounded-2xl text-center font-black text-xs uppercase tracking-widest text-slate-900 hover:bg-slate-900 hover:text-white transition-all italic shadow-sm">
                        <?php esc_html_e('Join the 500+ Experts', 'autoquill'); ?>
                    </a>
                </div>

                <div class="aq-plan-card bg-indigo-950 rounded-[2.5rem] p-8 md:p-10 flex flex-col shadow-2xl relative overflow-hidden border border-indigo-900">
                    <div class="absolute top-0 right-0 w-32 h-32 bg-indigo-500/10 rounded-full blur-3xl -mr-16 -mt-16"></div>

                    <div class="mb-8 relative z-10">
                    <span class="px-4 py-1.5 rounded-full bg-indigo-500/20 text-indigo-300 text-[10px] font-black uppercase tracking-widest italic border border-indigo-400/20">
                        <?php esc_html_e('Member Access', 'autoquill'); ?>
                    </span>
                        <h3 class="text-3xl font-black text-white mt-6 mb-2 uppercase italic tracking-tighter">
                            <?php esc_html_e('Connect System', 'autoquill'); ?>
                        </h3>
                    </div>

                    <div class="flex-grow relative z-10">
                        <?php if ($current_token): ?>
                            <div class="bg-indigo-900/40 rounded-3xl p-6 mb-8 border border-indigo-800/50">
                                <span class="text-[10px] font-black text-indigo-400 uppercase tracking-widest block mb-2 italic"><?php esc_html_e('Logged in as', 'autoquill'); ?></span>
                                <div class="text-lg font-bold text-white truncate italic"><?php echo esc_html($active_email); ?></div>
                                <div class="mt-4 pt-4 border-t border-indigo-800/50 flex justify-between items-center">
                                    <span class="text-[10px] font-black text-indigo-400 uppercase tracking-widest italic"><?php esc_html_e('Plan Status', 'autoquill'); ?></span>
                                    <span class="text-xs font-black text-emerald-400 uppercase italic tracking-tighter"><?php echo $this->is_pro() === 'pro' ? '💎 Pro' : 'Free' ?></span>
                                </div>
                            </div>
                            <button onclick="redirectToDashboard()" class="w-full py-5 bg-white text-indigo-950 rounded-2xl font-black text-xs uppercase tracking-widest hover:bg-indigo-50 transition-all shadow-xl italic mb-4">
                                <?php esc_html_e('Enter Workspace', 'autoquill'); ?>
                            </button>
                            <button onclick="handleLogout()" class="w-full text-indigo-400 hover:text-white font-black text-[10px] uppercase tracking-[0.2em] transition-all italic"><?php esc_html_e('Disconnect Account', 'autoquill'); ?></button>

                        <?php else: ?>
                            <div class="flex bg-white/5 p-1 rounded-2xl mb-6">
                                <button onclick="switchTab('quick')" id="tab-quick" class="tab-btn active flex-1 py-3 rounded-xl font-black text-[9px] uppercase tracking-widest transition-all italic">Quick</button>
                                <button onclick="switchTab('manual')" id="tab-manual" class="tab-btn flex-1 py-3 rounded-xl font-black text-[9px] uppercase tracking-widest transition-all text-indigo-300/50 italic">Manual</button>
                            </div>

                            <div id="quick-section" class="space-y-4">
                                <div id="step-email-confirm" class="space-y-4">
                                    <div class="bg-white/5 border border-white/10 p-5 rounded-2xl focus-within:border-indigo-400 transition-all">
                                        <input id="aq_email" type="email" value="<?php echo esc_attr($user_email); ?>" class="bg-transparent w-full font-bold text-white outline-none text-base placeholder:text-indigo-300/30" placeholder="your@email.com">
                                    </div>
                                    <button type="button" onclick="openUserModal()" class="w-full text-indigo-400 hover:text-white font-black text-[9px] uppercase tracking-widest italic transition-all"><?php esc_html_e('Existing WP Users', 'autoquill'); ?></button>
                                    <button onclick="handleBtn(this, startQuickAuth)" class="w-full py-5 bg-indigo-600 text-white rounded-2xl font-black text-xs uppercase tracking-widest shadow-xl hover:bg-indigo-500 transition-all italic">
                                        <?php esc_html_e('Get Access Code', 'autoquill'); ?>
                                    </button>
                                </div>

                                <div id="step-otp-entry" class="hidden space-y-4">
                                    <p class="text-green-500 text-[10px] font-medium text-center uppercase tracking-widest italic mb-2">
                                        <?php esc_html_e('A verification code has been sent to your email address.', 'autoquill'); ?>
                                    </p>

                                    <input id="aq_otp" type="text" maxlength="6" placeholder="000000" class="w-full text-center text-4xl font-black py-5 bg-white/5 rounded-2xl border-2 border-white/10 focus:border-indigo-400 outline-none transition text-white tracking-[0.2em]">

                                    <button onclick="handleBtn(this, verifyQuickAuth)" class="w-full py-5 bg-white text-indigo-950 rounded-2xl font-black text-xs uppercase tracking-widest italic flex items-center justify-center gap-3">
                                        <span><?php esc_html_e('Verify & Connect', 'autoquill'); ?></span>
                                    </button>
                                </div>
                            </div>

                            <div id="manual-section" class="hidden space-y-4">
                                <textarea id="aq_manual_token" rows="4" placeholder="Bearer eyJ..." class="w-full p-4 bg-white/5 rounded-2xl border border-white/10 text-white font-mono text-[10px] outline-none focus:border-indigo-400"></textarea>
                                <button onclick="handleBtn(this, connectManual)" class="w-full py-5 bg-white text-indigo-950 rounded-2xl font-black text-xs uppercase tracking-widest italic italic">
                                    <?php esc_html_e('Connect', 'autoquill'); ?>
                                </button>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div id="aq-msg" class="mt-4 text-center text-red-400 font-black text-[9px] uppercase tracking-widest py-3 rounded-xl bg-red-950/20 border border-red-900/30 hidden"></div>
                </div>
            </div>
        </div>

        <?php $this->render_footer(); ?>

        <div id="user-modal" style="display:none;" class="fixed inset-0 z-[9999] flex items-center justify-center p-4 bg-slate-950/90 backdrop-blur-sm">
            <div class="bg-white w-full max-w-xs rounded-[2.5rem] shadow-2xl overflow-hidden">
                <div class="p-6 bg-slate-50 border-b border-slate-100 text-center uppercase italic">
                    <h3 class="font-black text-slate-900 text-xs tracking-widest"><?php esc_html_e('Local Accounts', 'autoquill'); ?></h3>
                </div>
                <div class="p-4 max-h-60 overflow-y-auto">
                    <?php foreach ($wp_users as $user): ?>
                        <button type="button" onclick="setActiveUser('<?php echo esc_attr($user->user_email); ?>')" class="w-full p-4 mb-2 rounded-2xl border-2 border-slate-50 hover:border-indigo-500 hover:bg-indigo-50 transition-all text-left">
                            <div class="font-black text-slate-800 text-[10px] uppercase italic"><?php echo esc_html($user->display_name); ?></div>
                        </button>
                    <?php endforeach; ?>
                </div>
                <button type="button" onclick="closeUserModal()" class="w-full py-4 text-slate-400 font-black text-[9px] uppercase border-t border-slate-100 italic tracking-widest"><?php esc_html_e('Close', 'autoquill'); ?></button>
            </div>
        </div>

        <script>
            // JS LOGIC (Korundu & Düzenlendi)
            async function handleBtn(btn, callback) {
                const originalHtml = btn.innerHTML;
                btn.disabled = true;


                btn.innerHTML = `
        <div class="flex items-center justify-center gap-3">
            <div class="w-4 h-4 border-2 border-current border-t-transparent rounded-full animate-spin"></div>
            <span class="uppercase tracking-widest text-[10px] opacity-80">Processing...</span>
        </div>
    `;

                try {
                    await callback(btn);
                } catch (e) {
                    showError("Error!");
                    btn.innerHTML = originalHtml;
                } finally {
                    if (btn) {
                        btn.disabled = false;
                        btn.innerHTML = originalHtml;
                    }
                }
            }

            function switchTab(type) {
                const isQuick = type === 'quick';
                document.getElementById('tab-quick').classList.toggle('active', isQuick);
                document.getElementById('tab-manual').classList.toggle('active', !isQuick);
                document.getElementById('quick-section').classList.toggle('hidden', !isQuick);
                document.getElementById('manual-section').classList.toggle('hidden', isQuick);
            }

            async function startQuickAuth() {
                const email = document.getElementById('aq_email').value;
                const fd = new FormData();
                fd.append('action', 'aq_auth_action');
                fd.append('nonce', aq_vars.nonce);
                fd.append('step', 'register_and_otp');
                fd.append('email', email);
                const res = await fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', { method: 'POST', body: fd });
                const data = await res.json();
                if(data.success) {
                    document.getElementById('step-email-confirm').classList.add('hidden');
                    document.getElementById('step-otp-entry').classList.remove('hidden');
                } else { showError(data.data.message); }
            }

            async function verifyQuickAuth() {
                const otp = document.getElementById('aq_otp').value;
                const fd = new FormData();
                fd.append('action', 'aq_auth_action');
                fd.append('nonce', aq_vars.nonce);
                fd.append('step', 'verify_otp');
                fd.append('otp', otp);
                const res = await fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', { method: 'POST', body: fd });
                const data = await res.json();
                if(data.success) { await finalConnectAuto(data.data.temp_token); }
                else { showError("Invalid OTP."); }
            }

            async function finalConnectAuto(token) {
                const fd = new FormData();
                fd.append('action', 'aq_auth_action');
                fd.append('nonce', aq_vars.nonce);
                fd.append('step', 'final_connect');
                fd.append('token', token);
                const res = await fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', { method: 'POST', body: fd });
                const data = await res.json();
                if(data.success) { window.location.href = '<?php echo esc_url(admin_url('admin.php?page=api-data')); ?>'; }
            }

            async function connectManual() {
                const token = document.getElementById('aq_manual_token').value.trim();
                if(!token) return showError("Enter token");
                await finalConnectAuto(token);
            }

            async function handleLogout() {
                if(!confirm("Logout?")) return;
                const fd = new FormData();
                fd.append('action', 'aq_auth_action');
                fd.append('nonce', aq_vars.nonce);
                fd.append('step', 'logout');
                await fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', { method: 'POST', body: fd });
                window.location.reload();
            }

            function openUserModal() { document.getElementById('user-modal').style.display = 'flex'; }
            function closeUserModal() { document.getElementById('user-modal').style.display = 'none'; }
            function setActiveUser(email) { document.getElementById('aq_email').value = email; closeUserModal(); }
            function showError(msg) { const el = document.getElementById('aq-msg'); el.innerText = msg; el.classList.remove('hidden'); }
            function redirectToDashboard() { window.location.href = '<?php echo esc_url(admin_url('admin.php?page=api-data')); ?>'; }
        </script>
        <?php
    }

    public function render_footer() {
        $logo_url = plugins_url('plusclouds-logo.svg', __FILE__);
        ?>
        <footer class="mt-16 flex flex-col items-center justify-center space-y-4">
        <span class="text-[10px] uppercase tracking-[0.2em] text-slate-300 font-medium">
            <?php esc_html_e('Crafted by', 'autoquill'); ?>
        </span>
            <a href="https://plusclouds.com"
               target="_blank"
               rel="noopener"
               class="transition-all duration-300  opacity-40 hover:opacity-100">
                <img src="<?php echo esc_url($logo_url); ?>"
                     alt="PlusClouds"
                     class="h-4 w-auto">
            </a>
        </footer>
        <?php
    }

    public function aq_handle_auth_api() {
        check_ajax_referer('aq_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Unauthorized']);

        if (ob_get_length()) ob_clean();

        $step = isset($_POST['step']) ? sanitize_text_field(wp_unslash($_POST['step'])) : '';

        $leo_url = "https://apiv4.plusclouds.com";
        $client_id = "8dc1e611-5724-4f85-8464-34a9140b2eaf";
        $redirect_uri = "https://cloud.plusclouds.com/auth-callback";

        try {
            switch ($step) {
                case 'register_and_otp':
                    $email = sanitize_email(wp_unslash($_POST['email'] ?? ''));
                    if (empty($email)) throw new Exception('Email is required.');

                    // 1. Register işlemi
                    wp_remote_post("$leo_url/public/onboarding/register", [
                            'body' => ['email' => $email, 'partner_code' => 'autoquill', 'source' => 'COMMUNITY']
                    ]);

                    // 2. OAuth Session oluşturma
                    $session_params = http_build_query([
                            'client_id' => $client_id,
                            'response_type' => 'token',
                            'redirect_uri' => $redirect_uri
                    ]);

                    $session_res = wp_remote_get("$leo_url/iam/authentication/oauth/session?$session_params");
                    $session_data = json_decode(wp_remote_retrieve_body($session_res), true);

                    if (!empty($session_data['data']['session_id'])) {
                        $s_id = $session_data['data']['session_id'];
                        set_transient('aq_oauth_session', $s_id, 30 * MINUTE_IN_SECONDS);

                        // OTP mekanizmalarını tetikle
                        wp_remote_get("$leo_url/iam/authentication/oauth/$s_id/login-mechanisms?email=" . urlencode($email));
                        wp_remote_get("$leo_url/iam/authentication/oauth/$s_id/send-otp-email");

                        wp_send_json_success(['message' => 'OTP Sent']);
                    }
                    throw new Exception('Auth session could not be initialized.');
                    break;
                case 'logout':
                    delete_option('aq_token');
                    delete_option('aq_user_plan');
                    wp_send_json_success(['message' => 'Logged out']);
                    break; // Bu break çok önemli!
                case 'verify_otp':
                    $otp = sanitize_text_field(wp_unslash($_POST['otp'] ?? ''));
                    $s_id = get_transient('aq_oauth_session');
                    if (!$s_id) throw new Exception('Session expired, please try again.');

                    // OTP Doğrulama
                    $validate = wp_remote_post("$leo_url/iam/authentication/oauth/$s_id/validate-otp-email", [
                            'body' => ['password' => $otp]
                    ]);
                    $val_data = json_decode(wp_remote_retrieve_body($validate), true);

                    if (isset($val_data['data']) && $val_data['data'] === true) {
                        // Auth Code al
                        $auth_code_res = wp_remote_post("$leo_url/iam/authentication/oauth/$s_id/auth-code");
                        $auth_code = wp_remote_retrieve_body($auth_code_res);

                        // Access Token al (API v4 üzerinden)
                        $token_res = wp_remote_post("$leo_url/iam/authentication/oauth/$client_id/access-token", [
                                'body' => ['code' => $auth_code]
                        ]);
                        $token_data = json_decode(wp_remote_retrieve_body($token_res), true);

                        wp_send_json_success([
                                'temp_token' => $token_data['access_token'] ?? ''
                        ]);
                    }
                    throw new Exception('Invalid OTP code.');
                    break;

                case 'final_connect':
                    $token = sanitize_text_field(wp_unslash($_POST['token'] ?? ''));

                    if (empty($token)) throw new Exception('Token is missing.');

                    $check_res = wp_remote_get("$leo_url/leo/security/am-i-logged-in", [
                        'headers' => [
                            'Authorization' => 'Bearer ' . $token,
                            'Accept'        => 'application/json',
                        ],
                        'timeout' => 20,
                    ]);

                    if (is_wp_error($check_res)) throw new Exception('API Connection Error');

                    $status_code = wp_remote_retrieve_response_code($check_res);
                    $user_body   = json_decode(wp_remote_retrieve_body($check_res), true);

                    if ($status_code !== 200 || (!isset($user_body['logged_in']) && !isset($user_body['id']))) {
                        throw new Exception('Invalid Token: Please login again.');
                    }

                    $accounts_res = wp_remote_get("$leo_url/autoquill/accounts", [
                        'headers' => [
                            'Authorization' => 'Bearer ' . $token,
                            'Accept'        => 'application/json',
                        ],
                        'timeout' => 10,
                    ]);

                    $is_pro = 0;
                    if (!is_wp_error($accounts_res)) {
                        $accounts_body = json_decode(wp_remote_retrieve_body($accounts_res), true);
                        $package       = $accounts_body['data'][0]['package'] ?? 'FREE';
                        $is_pro        = $package !== 'FREE' ? 1 : 0;
                    }

                    delete_option('aq_bearer_token');
                    update_option('aq_token', $token);
                    update_option('aq_user_plan', $is_pro);

                    wp_send_json_success([
                        'message' => 'Connected!',
                        'plan'    => $is_pro ? 'pro' : 'free',
                        'email'   => $user_body['email'] ?? '',
                    ]);
                    break;

                default:
                    throw new Exception('Invalid action step.');
            }
        } catch (Exception $e) {
            // Hata mesajını JavaScript'in okuyabileceği şekilde dön
            wp_send_json_error(['message' => $e->getMessage()]);
        }
        wp_die();
    }

    public function ajax_preview() {
        check_ajax_referer('aq_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Unauthorized']);
        $res = wp_remote_get($this->preview_api_url . sanitize_text_field(wp_unslash($_POST['post_id'] ?? '')), ['headers' => ['Authorization' => 'Bearer ' . $this->token]]);

        if (is_wp_error($res)) {
            wp_send_json_error(['message' => $res->get_error_message()]);
            return;
        }

        $body = json_decode(wp_remote_retrieve_body($res), true);

        if (empty($body['data'])) {
            wp_send_json_error(['message' => 'No data returned from API.']);
            return;
        }

        wp_send_json_success($body['data']);
    }

    public function render_workspace() {
        $id = sanitize_text_field(wp_unslash($_GET['id'] ?? ''));
        $is_pro = $this->is_pro();
        echo '<style>
    html { scroll-behavior: smooth; } 
    .premium-gradient { background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%); }
    .seo-badge { background: rgba(255,255,255,0.1); backdrop-filter: blur(4px); }
</style>';
        ?>
        <?php $this->render_plan_badge(); ?>
        <div class="flex h-screen bg-slate-100 overflow-hidden font-sans">
            <div class="w-1/2 flex flex-col border-r border-slate-200 bg-white">
                <div class="p-5 border-b bg-slate-50 flex justify-between items-center">
                    <div class="flex flex-col">
                        <span class="font-extrabold text-slate-400 text-[10px] uppercase tracking-widest"><?php esc_html_e('Reference Source', 'autoquill'); ?></span>
                        <span class="text-[9px] text-indigo-500 font-bold"><?php esc_html_e('SEO Optimized Context Analysis', 'autoquill'); ?></span>
                    </div>
                    <a href="?page=api-data" class="text-indigo-600 font-extrabold text-sm hover:underline">← <?php esc_html_e('Back to AutoQuill', 'autoquill'); ?></a>
                </div>
                <div id="aq-src" class="flex-1 overflow-y-auto p-12 bg-white prose prose-slate max-w-none prose-p:text-slate-700 prose-p:text-lg prose-p:font-medium prose-headings:font-extrabold prose-headings:text-indigo-950 prose-headings:tracking-tight prose-li:text-slate-700 prose-li:text-lg prose-img:rounded-[2rem] prose-img:shadow-lg prose-strong:font-extrabold prose-strong:text-slate-900">
                    <div class="flex flex-col items-center justify-center h-full space-y-4 opacity-40">
                        <svg class="w-12 h-12 animate-spin text-indigo-600" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        <p class="font-extrabold text-sm uppercase tracking-widest text-slate-400"><?php esc_html_e('Fetching reference data...', 'autoquill'); ?></p>
                    </div>
                </div>
            </div>

            <div class="w-1/2 flex flex-col shadow-2xl z-10 overflow-hidden">
                <div class="p-8 bg-indigo-950 rounded-lg text-white relative overflow-hidden">
                    <div class="relative z-10">
                        <div class="flex justify-between items-center mb-6">
                            <span class="font-bold text-[10px] uppercase tracking-[0.2em] text-indigo-200/80"><?php esc_html_e('AI Engine v4.0', 'autoquill'); ?></span>
                            <div class="flex items-center">
                                <?php if($is_pro): ?>
                                    <span class="bg-indigo-950/10 text-indigo-300 border border-indigo-400/20 px-2 py-0.5 rounded-sm text-[9px] font-bold tracking-[0.2em] uppercase"><?php esc_html_e('💎 PRO', 'autoquill'); ?></span>
                                <?php else: ?>
                                    <a href="?page=plans" class="text-white border border-white/20 hover:bg-white hover:text-indigo-950 px-4 py-1.5 rounded-full text-[9px] font-black transition-all tracking-widest uppercase italic"><?php esc_html_e('UPGRADE', 'autoquill'); ?></a>
                                <?php endif; ?>
                            </div>
                        </div>

                        <h2 class="text-3xl text-white tracking-tighter uppercase mb-2 italic font-black"><?php esc_html_e('Human-Like SEO Content', 'autoquill'); ?></h2>
                        <p class="text-[11px] text-indigo-200 font-bold leading-relaxed uppercase tracking-wide max-w-lg italic"><?php esc_html_e('Elevate your site with Google SEO-focused, human-like and original content architecture.', 'autoquill'); ?></p>
                    </div>
                    <div class="absolute top-0 right-0 -mr-12 -mt-12 w-48 h-48 bg-indigo-500/10 rounded-full blur-3xl"></div>
                </div>

                <div class="p-8 bg-white border-b border-slate-100 grid grid-cols-2 sm:grid-cols-3 gap-6 shadow-sm">
                    <div class="space-y-2">
                        <p class="text-[10px] font-extrabold text-slate-400 uppercase tracking-widest"><?php esc_html_e('Writing Tone', 'autoquill'); ?></p>
                        <select id="tone" class="w-full border-slate-200 rounded-xl p-3 font-bold text-slate-700 focus:ring-2 focus:ring-indigo-500 transition appearance-none bg-slate-50">
                            <option value="formal"><?php esc_html_e('Formal & Professional', 'autoquill'); ?></option>
                            <option value="casual"><?php esc_html_e('Casual & Friendly', 'autoquill'); ?></option>
                        </select>
                    </div>
                    <div class="space-y-2">
                        <p class="text-[10px] font-extrabold text-slate-400 uppercase tracking-widest"><?php esc_html_e('Target Length', 'autoquill'); ?></p>
                        <input type="number" id="len" value="300" min="100" max="500"
                               class="w-full border-slate-200 rounded-xl p-3 font-bold text-slate-700 focus:ring-2 focus:ring-indigo-500 transition bg-slate-50"
                               oninput="if(value>500)value=500">
                    </div>
                    <div class="space-y-2 relative">
                        <p class="text-[10px] font-extrabold text-slate-400 uppercase tracking-widest"><?php esc_html_e('Output Language', 'autoquill'); ?></p>
                        <select id="lang" <?php echo !$is_pro ? 'disabled' : ''; ?> class="w-full border-slate-200 rounded-xl p-3 font-bold text-slate-700 focus:ring-2 focus:ring-indigo-500 transition <?php echo !$is_pro ? 'bg-slate-100 opacity-60 cursor-not-allowed' : 'bg-slate-50'; ?>">
                            <option value="tr">Türkçe</option>
                            <option value="en">English (US)</option>
                            <option value="ar">العربية (Arabic)</option>
                            <option value="fr">Français</option>
                            <option value="nl">Nederlands</option>
                        </select>
                        <?php if(!$is_pro): ?>
                            <a href="?page=plans" class="absolute -bottom-5 left-0 text-[8px] font-black text-indigo-600 uppercase underline decoration-indigo-300 hover:text-indigo-950 transition-all"><?php esc_html_e('Unlock 5+ Languages in Pro', 'autoquill'); ?></a>
                        <?php endif; ?>
                    </div>

                    <div class="col-span-3 flex gap-8 pt-6 border-t border-slate-50 items-center mt-2">
                        <label class="flex items-center gap-3 font-extrabold text-sm text-slate-600 cursor-pointer group">
                            <input type="checkbox" id="img" checked class="w-5 h-5 rounded-lg text-indigo-600 focus:ring-indigo-500 border-slate-300">
                            <span class="group-hover:text-indigo-600 transition-colors"><?php esc_html_e('Include AI Images', 'autoquill'); ?></span>
                        </label>

                        <div class="flex flex-col relative group">
                            <label class="flex items-center gap-3 font-extrabold text-sm <?php echo !$is_pro ? 'text-slate-400 cursor-not-allowed' : 'text-slate-600 cursor-pointer'; ?>">
                                <input type="checkbox" id="aff" <?php echo !$is_pro ? 'checked disabled' : 'checked'; ?> class="w-5 h-5 rounded-lg text-indigo-600 focus:ring-indigo-500 border-slate-300">
                                <span><?php esc_html_e('Affiliate Links', 'autoquill'); ?></span>
                            </label>
                            <?php if(!$is_pro): ?>
                                <a href="?page=plans" class="text-[8px] font-black text-amber-600 uppercase tracking-tighter mt-1 hover:underline decoration-amber-300"><?php esc_html_e('Ads enabled (Go Pro to remove)', 'autoquill'); ?></a>
                            <?php endif; ?>
                        </div>

                        <button id="gen-btn" class="ml-auto hover:bg-indigo-700 text-white bg-indigo-900 px-12 py-5 rounded-2xl font-black shadow-xl shadow-indigo-200 hover:shadow-indigo-400 transition-all active:scale-95 flex items-center gap-3">
                            <span class="text-lg uppercase tracking-tighter"><?php esc_html_e('Start Generation', 'autoquill'); ?></span>
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M13 10V3L4 14h7v7l9-11h-7z"/></svg>
                        </button>
                    </div>
                </div>

                <div class="flex-1 overflow-y-auto p-10 bg-slate-50">
                    <div id="aq-placeholder" class="h-full border-4 border-dashed border-slate-200 rounded-[3rem] flex flex-col items-center justify-center text-slate-300 bg-white/50 space-y-4">
                        <div class="bg-slate-100 p-6 rounded-full">
                            <svg class="w-16 h-16 opacity-20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10l4 4v10a2 2 0 01-2 2zM14 4v4h4"/></svg>
                        </div>
                        <div class="text-center">
                            <p class="font-black italic text-xl text-slate-400 uppercase tracking-tighter"><?php esc_html_e('Waiting for Magic', 'autoquill'); ?></p>
                            <p class="text-xs font-bold text-slate-400 mt-1"><?php esc_html_e('Configure your settings and click Generate', 'autoquill'); ?></p>
                        </div>
                    </div>

                    <div id="ai-preview-container" class="hidden space-y-6">
                        <div id="seo-meta-box" class="bg-white border border-slate-200 rounded-[2rem] p-8 shadow-sm space-y-4">
                            <div class="flex items-center gap-2 mb-2">
                            <span class="p-1.5 bg-indigo-100 text-indigo-600 rounded-lg">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"/></svg>
                            </span>
                                <h3 class="font-black text-slate-900 text-xs uppercase tracking-widest "><?php esc_html_e('SEO Optimization Pack', 'autoquill'); ?></h3>
                            </div>
                            <div class="grid grid-cols-1 gap-4">
                                <div>
                                    <label class="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] block mb-1"><?php esc_html_e('Meta Title', 'autoquill'); ?></label>
                                    <p id="pv-meta-title" class="text-sm font-bold text-slate-800"></p>
                                </div>
                                <div>
                                    <label class="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] block mb-1"><?php esc_html_e('Meta Description', 'autoquill'); ?></label>
                                    <p id="pv-meta-desc" class="text-xs font-medium text-slate-500 leading-relaxed"></p>
                                </div>
                                <div class="flex gap-4 border-t border-slate-50 pt-4">
                                    <div class="flex-1">
                                        <label class="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] block mb-1"><?php esc_html_e('Keywords', 'autoquill'); ?></label>
                                        <p id="pv-meta-keys" class="text-[10px] font-bold text-indigo-600 uppercase "></p>
                                    </div>
                                    <div class="flex-1">
                                        <label class="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] block mb-1"><?php esc_html_e('Locale', 'autoquill'); ?></label>
                                        <p id="pv-locale" class="text-[10px] font-bold text-slate-800 uppercase "></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="bg-white rounded-[3rem] shadow-2xl border border-slate-100 overflow-hidden mb-6 animate-in fade-in slide-in-from-bottom-8 duration-700">
                            <div class="premium-gradient p-14 text-white text-center relative">
                                <h2 id="ai-pv-title" class="text-4xl text-white m-0 leading-tight italic tracking-tighter font-black"></h2>
                                <div class="mt-4 inline-flex items-center gap-2 bg-white/10 px-4 py-1.5 rounded-full text-[10px] font-bold uppercase tracking-widest border border-white/20 italic">
                                    <span class="w-2 h-2 bg-emerald-400 rounded-full animate-pulse"></span>
                                    <?php esc_html_e('AI Generated Content Ready', 'autoquill'); ?>
                                </div>
                            </div>
                            <div id="ai-pv-body" class="p-16 prose prose-indigo max-w-none text-slate-700 prose-headings:font-black prose-h2:text-indigo-950 prose-h2:italic prose-h2:tracking-tighter prose-a:text-indigo-600 prose-table:border-2 prose-table:rounded-3xl prose-img:rounded-[2.5rem] prose-img:shadow-2xl"></div>

                            <div class="p-12 bg-slate-50 border-t border-slate-100 space-y-4">
                                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                    <input type="hidden" name="action" value="save_api_post">
                                    <input type="hidden" name="blog_post_id" value="<?php echo esc_attr($id); ?>">
                                    <input type="hidden" name="is_ai" value="1">
                                    <textarea name="ai_json_data" id="aq-json-input" class="hidden"></textarea>
                                    <?php wp_nonce_field('aq_save_post', 'aq_save_nonce'); ?>

                                    <?php if (!$is_pro): ?>
                                    <p class="text-[9px] text-amber-700 font-medium text-center leading-relaxed px-3 py-2.5 bg-amber-50 rounded-2xl mb-3">
                                        <?php esc_html_e('License Notice: Free usage requires active affiliate links. Removing or modifying links in free-tier content is a violation of terms and may lead to permanent account suspension.', 'autoquill'); ?>
                                    </p>
                                    <?php endif; ?>
                                    <button type="submit" class="w-full bg-emerald-500 text-white font-black py-5 rounded-[2rem] text-lg shadow-2xl shadow-emerald-200 hover:bg-emerald-600 hover:-translate-y-1 transition transform tracking-tighter uppercase italic">
                                        <?php esc_html_e('Publish to WordPress', 'autoquill'); ?>
                                    </button>
                                </form>

                                <button id="download-json-btn" class="w-full bg-slate-900 text-white font-black py-4 rounded-[2rem] text-sm hover:bg-slate-800 transition tracking-widest uppercase italic flex items-center justify-center gap-2">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a2 2 0 002 2h12a2 2 0 002-2v-1m-4-4l-4 4m0 0l-4-4m4 4V4"/></svg>
                                    <?php esc_html_e('Download Content JSON', 'autoquill'); ?>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <script>
            jQuery(document).ready(function($) {
                const pid = "<?php echo esc_attr($id); ?>";
                const isProUser = <?php echo $is_pro ? 'true' : 'false'; ?>;
                let currentAiJson = null;

                $.post(aq_vars.ajax_url, {action:'fetch_post_preview', nonce:aq_vars.nonce, post_id:pid}, function(r){
                    if(r.success && r.data) {
                        const htmlContent = r.data.body ? marked.parse(r.data.body) : '';
                        $('#aq-src').html('<h1 class="font-black text-indigo-950 italic tracking-tighter text-4xl mb-8">' + (r.data.title || '') + '</h1>' + htmlContent);
                    }
                });

                $('#gen-btn').click(function(){
                    const $btn = $(this);
                    $btn.prop('disabled',true).addClass('opacity-50');
                    $btn.find('span').text("<?php esc_html_e('Generating...', 'autoquill'); ?>");

                    $.post(aq_vars.ajax_url, {
                        action: 'generate_ai_content',
                        nonce: aq_vars.nonce,
                        post_id: pid,
                        tone: $('#tone').val(),
                        max_length: $('#len').val(),
                        images: $('#img').is(':checked'),
                        affiliates: isProUser ? $('#aff').is(':checked') : true,
                        language: isProUser ? $('#lang').val() : 'tr'
                    }, function(r){
                        $btn.prop('disabled',false).removeClass('opacity-50');
                        $btn.find('span').text("<?php esc_html_e('Start Generation', 'autoquill'); ?>");

                        if(r.success){
                            currentAiJson = r.data; // Store full JSON

                            $('#aq-placeholder').hide();
                            $('#ai-preview-container').removeClass('hidden');

                            // Fill Metadata Preview
                            $('#pv-meta-title').text(r.data.meta_title || '');
                            $('#pv-meta-desc').text(r.data.meta_description || '');
                            $('#pv-meta-keys').text(r.data.meta_keywords || '');
                            $('#pv-locale').text(r.data.locale || 'us');

                            // Fill Content Preview
                            $('#ai-pv-title').text(r.data.title || "<?php esc_html_e('AI Generated Post', 'autoquill'); ?>");
                            const renderedHtml = marked.parse(r.data.body);
                            $('#ai-pv-body').html(renderedHtml);

                            // Prepare hidden JSON for WordPress Save (body HTML olarak gönderilir)
                            const saveData = Object.assign({}, r.data, { body: renderedHtml });
                            $('#aq-json-input').val(JSON.stringify(saveData));

                            $('.w-1\/2.shadow-2xl .flex-1').animate({ scrollTop: 0 }, 1000);
                        }
                    });
                });

                // JSON Download Logic
                $('#download-json-btn').click(function(e) {
                    e.preventDefault();
                    if(!currentAiJson) return;

                    const dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(currentAiJson, null, 4));
                    const downloadAnchorNode = document.createElement('a');
                    downloadAnchorNode.setAttribute("href",     dataStr);
                    downloadAnchorNode.setAttribute("download", "autoquill_" + pid + ".json");
                    document.body.appendChild(downloadAnchorNode);
                    downloadAnchorNode.click();
                    downloadAnchorNode.remove();
                });
            });
        </script>
        <?php
    }

    public function ajax_generate() {
        check_ajax_referer('aq_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Unauthorized']);

        // Arka plan Pro kontrolü
        $is_pro = $this->is_pro();

        // Dil: Pro değilse sadece 'tr' zorunlu
        $selected_lang = $is_pro && isset($_POST['language']) ? sanitize_text_field(wp_unslash($_POST['language'])) : 'tr';

        // Affiliates: Pro değilse her zaman true zorunlu
        $include_affiliates = $is_pro ? (isset($_POST['affiliates']) && $_POST['affiliates'] === 'true') : true;

        // Max Length: Temizlik ve Sınırlandırma
        $max_length = isset($_POST['max_length']) ? intval($_POST['max_length']) : 300;
        if ($max_length > 500) {
            $max_length = 500;
        } elseif ($max_length <= 0) {
            $max_length = 300;
        }

        $res = wp_remote_post($this->generate_api_url, [
                'headers' => [
                        'Authorization' => 'Bearer ' . $this->token,
                        'Content-Type'  => 'application/json'
                ],
                'body'    => json_encode([
                        "blog_post_id"       => sanitize_text_field(wp_unslash($_POST['post_id'] ?? '')),
                        "include_affiliates" => $include_affiliates,
                        "language"           => $selected_lang,
                        "include_images"     => isset($_POST['images']) && $_POST['images'] === 'true',
                        "max_length"         => $max_length,
                        "tone"               => sanitize_text_field(wp_unslash($_POST['tone'] ?? ''))
                ]),
                'timeout' => 120
        ]);

        if (is_wp_error($res)) {
            wp_send_json_error(['message' => $res->get_error_message()]);
            return;
        }

        $body = json_decode(wp_remote_retrieve_body($res), true);
        wp_send_json_success($body['data'] ?? $body);
    }

    public function handle_settings() {
        check_admin_referer('aq_save_settings', 'aq_settings_nonce');
        if (!current_user_can('manage_options')) wp_die(esc_html__('Unauthorized', 'autoquill'));

        if (isset($_POST['aq_token'])) {
            update_option('aq_token', sanitize_text_field(wp_unslash($_POST['aq_token'])));
        }
        wp_safe_redirect(admin_url('admin.php?page=api-data'));
        exit;
    }

    public function handle_save_draft() {
        check_admin_referer('aq_save_post', 'aq_save_nonce');
        if (!current_user_can('manage_options')) wp_die(esc_html__('Unauthorized', 'autoquill'));

        $id = sanitize_text_field(wp_unslash($_POST['blog_post_id'] ?? ''));
        $is_ai = isset($_POST['is_ai']);

        $res = wp_remote_get($this->preview_api_url . $id, [
                'headers' => ['Authorization' => 'Bearer ' . $this->token]
        ]);
        $api_meta = json_decode(wp_remote_retrieve_body($res), true)['data'];

        if ($is_ai && !empty($_POST['ai_json_data'])) {
            $ai_data = json_decode(sanitize_text_field(wp_unslash($_POST['ai_json_data'])), true);

            $post_title   = sanitize_text_field($ai_data['title']);
            $post_content = wp_kses_post($ai_data['body']);
            $post_excerpt = wp_kses_post($ai_data['abstract']); // Abstract özet alanına

            $meta_title   = sanitize_text_field($ai_data['meta_title']);
            $meta_desc    = sanitize_text_field($ai_data['meta_description']);
            $meta_keys    = sanitize_text_field($ai_data['meta_keywords']);
            $locale       = sanitize_text_field($ai_data['locale']);
            $inspired_by  = sanitize_text_field($ai_data['inspired_by']);
        } else {
            $post_title   = sanitize_text_field($api_meta['title']);
            $post_content = wp_kses_post($api_meta['body']);
            $post_excerpt = '';
            $meta_title   = '';
            $meta_desc    = '';
            $meta_keys    = '';
            $locale       = '';
            $inspired_by  = '';
        }

        $post_id = wp_insert_post([
                'post_title'   => $post_title,
                'post_content' => $post_content,
                'post_excerpt' => $post_excerpt,
                'post_status'  => 'draft',
                'post_author'  => get_current_user_id()
        ]);

        if ($post_id) {
            update_post_meta($post_id, '_aq_meta_title', $meta_title);
            update_post_meta($post_id, '_aq_meta_description', $meta_desc);
            update_post_meta($post_id, '_aq_meta_keywords', $meta_keys);
            update_post_meta($post_id, '_aq_locale', $locale);
            update_post_meta($post_id, '_aq_inspired_by', $inspired_by);

            if (defined('WPSEO_VERSION')) {
                update_post_meta($post_id, '_yoast_wpseo_title', $meta_title);
                update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_desc);
                update_post_meta($post_id, '_yoast_wpseo_focuskw', $meta_keys);
            }

            if (!empty($api_meta['header_image'])) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                require_once ABSPATH . 'wp-admin/includes/media.php';
                require_once ABSPATH . 'wp-admin/includes/image.php';
                $tmp = download_url($api_meta['header_image']);
                if (!is_wp_error($tmp)) {
                    $file = ['name' => basename($api_meta['header_image']), 'tmp_name' => $tmp];
                    $aid = media_handle_sideload($file, $post_id);
                    if (!is_wp_error($aid)) set_post_thumbnail($post_id, $aid);
                    wp_delete_file($tmp);
                }
            }
        }

        wp_safe_redirect(admin_url('edit.php?post_status=draft'));
        exit;
    }

    public function is_pro() {
        return (int) get_option('aq_user_plan', 0);
    }

    public function render_plan_badge() {
        $is_pro = $this->is_pro();
        $has_token = !empty(get_option('aq_token', ''));
        if (!$has_token) return;
        ?>
        <?php if ($is_pro): ?>
            <div class="flex items-center justify-end px-6 pt-4 pb-0 mb-5">
                <div class="relative group">
                    <div class="absolute -inset-0.5 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-full blur opacity-30 group-hover:opacity-60 transition duration-1000"></div>

                    <div class="relative inline-flex items-center gap-2.5 bg-indigo-950 border border-indigo-400/30 text-indigo-50 px-5 py-2 rounded-full shadow-2xl backdrop-blur-md">
                        <div class="flex items-center justify-center">
                            <span class="relative flex h-2 w-2">
                                <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-indigo-400 opacity-75"></span>
                                <span class="relative inline-flex rounded-full h-2 w-2 bg-indigo-500"></span>
                            </span>
                        </div>

                        <span class="text-[10px] font-black uppercase tracking-[0.25em] bg-gradient-to-r from-indigo-100 to-indigo-300 bg-clip-text text-transparent">
                            <?php esc_html_e("AutoQuill Pro Edition", 'autoquill'); ?>
                        </span>

                        <svg class="w-3.5 h-3.5 text-indigo-400" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87L18.18 22 12 18.27 5.82 22 7 14.14l-5-4.87 6.91-1.01L12 2z"></path>
                        </svg>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="px-6 pt-4 pb-0 mb-8">
                <div class="relative overflow-hidden flex flex-col lg:flex-row items-center justify-between bg-amber-50 border border-amber-200/60 rounded-[2rem] px-10 py-5 group transition-all duration-500 hover:shadow-xl hover:shadow-amber-900/5">

                    <div class="absolute -right-10 -top-10 w-48 h-48 bg-white rounded-full blur-3xl opacity-60"></div>
                    <div class="absolute left-0 top-0 w-2 h-full bg-amber-400"></div>

                    <div class="relative z-10 flex flex-col md:flex-row items-center gap-8 lg:gap-16">
                        <div>
                            <div class="flex items-center gap-2 mb-1">
                                <span class="text-[10px] font-black text-amber-600 uppercase tracking-[0.3em] italic"><?php esc_html_e('Limited Version', 'autoquill'); ?></span>
                                <div class="w-1 h-1 bg-amber-300 rounded-full"></div>
                                <span class="text-[10px] font-black text-indigo-950 uppercase tracking-[0.3em] italic"><?php esc_html_e('Pro Required', 'autoquill'); ?></span>
                            </div>
                            <h3 class="text-2xl font-black text-indigo-950 uppercase italic tracking-tighter leading-none">
                                <?php esc_html_e('Scale Your Content', 'autoquill'); ?> <span class="text-amber-600"><?php esc_html_e('To Pro', 'autoquill'); ?></span>
                            </h3>
                        </div>

                        <div class="flex flex-wrap justify-center md:justify-start items-center gap-6">
                            <div class="flex items-center gap-2">
                                <svg class="w-4 h-4 text-indigo-950/40" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M3 5h12M9 3v2m1.048 9.5a18.022 18.022 0 01-3.827-5.806m11.779 5.806c-1.127-3.236-3.21-6.267-5.806-8.812m1.056 12.812A15.312 15.312 0 0113 14m0 0a15.312 15.312 0 01-2.286-5.812M13 14a15.31 10.31 0 002.586-5.812M13 14l3 3m-3-3l-3 3"/></svg>
                                <span class="text-[10px] font-black text-indigo-950/70 uppercase italic tracking-widest"><?php esc_html_e('Global Languages', 'autoquill'); ?></span>
                            </div>
                            <div class="flex items-center gap-2 border-l border-amber-200 pl-6">
                                <svg class="w-4 h-4 text-indigo-950/40" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M13 10V3L4 14h7v7l9-11h-7z"/></svg>
                                <span class="text-[10px] font-black text-indigo-950/70 uppercase italic tracking-widest"><?php esc_html_e('Advanced Engine', 'autoquill'); ?></span>
                            </div>
                        </div>
                    </div>

                    <div class="relative z-10 mt-6 lg:mt-0">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=plans')); ?>" class="inline-flex hover:text-white items-center gap-4 bg-indigo-950 text-white px-10 py-4 rounded-2xl text-[11px] font-black uppercase tracking-[0.2em] hover:bg-black transition-all shadow-xl italic">
                            <span><?php esc_html_e('UPGRADE NOW', 'autoquill'); ?></span>
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path d="M13 7l5 5m0 0l-5 5m5-5H6" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                        </a>
                    </div>
                </div>
            </div>
        <?php endif;
    }
}
new API_Data_Fetcher();